########################################################################
#
# $Id: CPdetect.pm,v 1.14 2002/12/02 18:51:37 gosha Exp $
#
# Copyright (C) 2001-2002 Okunev Igor <gosha@prv.mts-nn.ru>
#
########################################################################
package Cyrillic::CPdetect;

use strict;
use vars qw( @ISA @EXPORT_OK $VERSION $DETECT_TYPE $COMPRESS_STR );
use Exporter;

($VERSION='$Revision: 1.14 $')=~s/^\S+\s+(\S+)\s+.*/$1/;

@ISA 			= qw( Exporter );
@EXPORT_OK		= qw( cp_detect );

$DETECT_TYPE	= 2;# 1 -    
					# 2 -    %TABLE

$COMPRESS_STR	= 1;# 1 - ''     (>3)
					#       (    )
					# 0 -  ...  ~  10%.

my %CHECK = (
		'KOI8-R'		=>	sub{ 1; },
		'ISO-8859-5'	=>	sub{$_[0]=~tr/\xA0\xA1\xB0\xB1\xB2\xB3\xB4\xB5\xB6\xB7\xB8\xB9\xBA\xBB\xBC\xBD\xBE\xBF\xC0\xC1\xC2\xC3\xC4\xC5\xC6\xC7\xC8\xC9\xCA\xCB\xCC\xCD\xCE\xCF\xD0\xD1\xD2\xD3\xD4\xD5\xD7\xD8\xD9\xDA\xDB\xDC\xDD\xDE\xDF\xE0\xE1\xE2\xE3\xE4\xE5\xE6\xE7\xE8\xE9\xEA\xEB\xEC\xED\xEE\xEF\xF1/\x9A\xB3\xE1\xE2\xF7\xE7\xE4\xE5\xF6\xFA\xE9\xEA\xEB\xEC\xED\xEE\xEF\xF0\xF2\xF3\xF4\xF5\xE6\xE8\xE3\xFE\xFB\xFD\xFF\xF9\xF8\xFC\xE0\xF1\xC1\xC2\xD7\xC7\xC4\xC5\xDA\xC9\xCA\xCB\xCC\xCD\xCE\xCF\xD0\xD2\xD3\xD4\xD5\xC6\xC8\xC3\xDE\xDB\xDD\xDF\xD9\xD8\xDC\xC0\xD1\xA3/},
		'IBM866'		=>	sub{$_[0]=~tr/\xFF\xF8\xFA\xF0\x80\x81\x82\x83\x84\x85\x86\x87\x88\x89\x8A\x8B\x8C\x8D\x8E\x8F\x90\x91\x92\x93\x94\x95\x96\x97\x98\x99\x9A\x9B\x9C\x9D\x9E\x9F\xA0\xA1\xA2\xA3\xA4\xA5\xA6\xA7\xA8\xA9\xAA\xAB\xAC\xAD\xAE\xAF\xE0\xE1\xE2\xE3\xE4\xE5\xE6\xE7\xE8\xE9\xEA\xEB\xEC\xED\xEE\xEF\xF1\xF9\xFB\xC4\xB3\xDA\xBF\xC0\xD9\xC3\xB4\xC2\xC1\xC5\xCD\xBA\xD5\xD6\xC9\xB8\xB7\xBB\xD4\xD3\xC8\xBE\xBD\xBC\xC6\xC7\xCC\xB5\xB6\xB9\xD1\xD2\xCB\xCF\xD0\xCA\xD8\xD7\xCE\xDF\xDC\xDB\xDD\xDE\xB0\xB1\xB2\xFE/\x9A\x9C\x9E\xB3\xE1\xE2\xF7\xE7\xE4\xE5\xF6\xFA\xE9\xEA\xEB\xEC\xED\xEE\xEF\xF0\xF2\xF3\xF4\xF5\xE6\xE8\xE3\xFE\xFB\xFD\xFF\xF9\xF8\xFC\xE0\xF1\xC1\xC2\xD7\xC7\xC4\xC5\xD6\xDA\xC9\xCA\xCB\xCC\xCD\xCE\xCF\xD0\xD2\xD3\xD4\xD5\xC6\xC8\xC3\xDE\xDB\xDD\xDF\xD9\xD8\xDC\xC0\xD1\xA3\x95\x96\x80\x81\x82\x83\x84\x85\x86\x87\x88\x89\x8A\xA0\xA1\xA2\xA4\xA5\xA6\xA7\xA8\xA9\xAA\xAB\xAC\xAD\xAE\xAF\xB0\xB1\xB2\xB4\xB5\xB6\xB7\xB8\xB9\xBA\xBB\xBC\xBD\xBE\x8B\x8C\x8D\x8E\x8F\x90\x91\x92\x94/},
		'CP1251'		=>	sub{$_[0]=~tr/\xA0\xA9\xB0\xB7\xA8\xC0\xC1\xC2\xC3\xC4\xC5\xC6\xC7\xC8\xC9\xCA\xCB\xCC\xCD\xCE\xCF\xD0\xD1\xD2\xD3\xD4\xD5\xD6\xD7\xD8\xD9\xDA\xDB\xDC\xDD\xDE\xDF\xE0\xE1\xE2\xE3\xE4\xE5\xE6\xE7\xE8\xE9\xEA\xEB\xEC\xED\xEE\xEF\xF0\xF1\xF2\xF3\xF4\xF5\xF6\xF7\xF8\xF9\xFA\xFB\xFC\xFD\xFE\xFF\xB8/\x9A\xBF\x9C\x9E\xB3\xE1\xE2\xF7\xE7\xE4\xE5\xF6\xFA\xE9\xEA\xEB\xEC\xED\xEE\xEF\xF0\xF2\xF3\xF4\xF5\xE6\xE8\xE3\xFE\xFB\xFD\xFF\xF9\xF8\xFC\xE0\xF1\xC1\xC2\xD7\xC7\xC4\xC5\xD6\xDA\xC9\xCA\xCB\xCC\xCD\xCE\xCF\xD0\xD2\xD3\xD4\xD5\xC6\xC8\xC3\xDE\xDB\xDD\xDF\xD9\xD8\xDC\xC0\xD1\xA3/},
		'MAC-UK'		=>	sub{$_[0]=~tr/\xCA\xA9\xA1\xD6\xDD\x80\x81\x82\x83\x84\x85\x86\x87\x88\x89\x8A\x8B\x8C\x8D\x8E\x8F\x90\x91\x92\x93\x94\x95\x96\x97\x98\x99\x9A\x9B\x9C\x9D\x9E\x9F\xE0\xE1\xE2\xE3\xE4\xE5\xE6\xE7\xE8\xE9\xEA\xEB\xEC\xED\xEE\xEF\xF0\xF1\xF2\xF3\xF4\xF5\xF6\xF7\xF8\xF9\xFA\xFB\xFC\xFD\xFE\xDF\xDE\xC3\xC5\xB2\xB3/\x9A\xBF\x9C\x9F\xB3\xE1\xE2\xF7\xE7\xE4\xE5\xF6\xFA\xE9\xEA\xEB\xEC\xED\xEE\xEF\xF0\xF2\xF3\xF4\xF5\xE6\xE8\xE3\xFE\xFB\xFD\xFF\xF9\xF8\xFC\xE0\xF1\xC1\xC2\xD7\xC7\xC4\xC5\xD6\xDA\xC9\xCA\xCB\xCC\xCD\xCE\xCF\xD0\xD2\xD3\xD4\xD5\xC6\xC8\xC3\xDE\xDB\xDD\xDF\xD9\xD8\xDC\xC0\xD1\xA3\x96\x97\x98\x99/},
		'IBM855'		=>	sub{$_[0]=~tr/\xFF\x85\xA1\xA3\xEC\xAD\xA7\xA9\xEA\xF4\xB8\xBE\xC7\xD1\xD3\xD5\xD7\xDD\xE2\xE4\xE6\xE8\xAB\xB6\xA5\xFC\xF6\xFA\x9F\xF2\xEE\xF8\x9D\xE0\xA0\xA2\xEB\xAC\xA6\xA8\xE9\xF3\xB7\xBD\xC6\xD0\xD2\xD4\xD6\xD8\xE1\xE3\xE5\xE7\xAA\xB5\xA4\xFB\xF5\xF9\x9E\xF1\xED\xF7\x9C\xDE\x84\xC4\xB3\xDA\xBF\xC0\xD9\xC3\xB4\xC2\xC1\xC5\xCD\xBA\xC9\xBB\xC8\xBC\xCC\xB9\xCB\xCA\xCE\xDF\xDC\xDB\xB0\xB1\xB2\xFE/\x9A\xB3\xE1\xE2\xF7\xE7\xE4\xE5\xF6\xFA\xE9\xEA\xEB\xEC\xED\xEE\xEF\xF0\xF2\xF3\xF4\xF5\xE6\xE8\xE3\xFE\xFB\xFD\xFF\xF9\xF8\xFC\xE0\xF1\xC1\xC2\xD7\xC7\xC4\xC5\xD6\xDA\xC9\xCA\xCB\xCC\xCD\xCE\xCF\xD0\xD2\xD3\xD4\xD5\xC6\xC8\xC3\xDE\xDB\xDD\xDF\xD9\xD8\xDC\xC0\xD1\xA3\x80\x81\x82\x83\x84\x85\x86\x87\x88\x89\x8A\xA0\xA1\xA5\xA8\xAB\xAE\xB1\xB5\xB8\xBB\xBE\x8B\x8C\x8D\x90\x91\x92\x94/}
	);

my %TABLE = (
				"\xD4\xD8",14932,"\xC9\xCA",11411,"\xD9\xCA",10931,"\xC1\xD4",9947,"\xD3\xD4",8527,"\xD7\xC1",8142,"\xCE\xD9",7825,"\xD3\xD1",7645,"\xD2\xC1",7438,"\xD2\xCF",6902,"\xCF\xD7",6417,"\xDB\xC9",6251,"\xC5\xCE",6031,"\xD0\xCF",5917,"\xC9\xD4",5767,"\xD2\xC5",5706,"\xCE\xC9",5560,"\xCB\xC1",5533,"\xD0\xD2",5488,"\xCF\xD3",5334,"\xC9\xD7",5305,"\xC5\xD2",5249,"\xD7\xDB",4991,"\xC1\xCE",4886,"\xD8\xD3",4542,"\xD2\xC9",4305,"\xC1\xD7",4284,"\xCE\xCE",4245,"\xDD\xC9",4234,"\xCF\xD2",4163,"\xDA\xC1",4054,"\xD7\xCF",3928,
				"\xCE\xC1",3876,"\xCB\xCF",3830,"\xCF\xD4",3692,"\xCE\xCF",3681,"\xCF\xCC",3680,"\xCC\xC5",3613,"\xD4\xC5",3545,"\xCF\xC4",3518,"\xCC\xC9",3465,"\xD4\xC9",3350,"\xCC\xCF",3239,"\xD7\xD9",3218,"\xD3\xCB",3174,"\xD4\xCF",3171,"\xC5\xD3",3128,"\xC5\xCC",3090,"\xC0\xDD",3054,"\xCC\xD8",3033,"\xD4\xC1",3033,"\xC1\xD3",2958,"\xCF\xC2",2912,"\xD4\xD2",2902,"\xC5\xD4",2886,"\xD7\xC5",2848,"\xCA\xD3",2842,"\xCF\xCE",2726,"\xC1\xCC",2694,"\xC9\xCE",2675,"\xCE\xC5",2647,"\xC1\xD2",2550,"\xC9\xD2",2501,"\xC9\xD3",2492,
				"\xC9\xC5",2485,"\xCC\xC1",2414,"\xCB\xC9",2401,"\xC9\xCB",2401,"\xD0\xC5",2398,"\xC1\xDA",2379,"\xCD\xC5",2274,"\xC4\xCF",2250,"\xDE\xC5",2198,"\xC1\xC0",2180,"\xC4\xC5",2170,"\xCF\xCB",2134,"\xC5\xD7",2039,"\xCF\xD0",2014,"\xDE\xC9",1967,"\xCF\xCD",1964,"\xD3\xCF",1959,"\xD7\xC9",1953,"\xC5\xCD",1948,"\xCD\xC1",1944,"\xCD\xCF",1916,"\xD5\xD4",1873,"\xC5\xC4",1839,"\xC4\xC9",1816,"\xC7\xCF",1772,"\xD0\xC1",1771,"\xD9\xD7",1752,"\xC9\xDA",1747,"\xCF\xC7",1716,"\xD2\xD5",1700,"\xC4\xC1",1624,"\xC1\xC4",1612,
				"\xCE\xD5",1610,"\xD3\xD0",1531,"\xCB\xD2",1528,"\xD4\xD7",1514,"\xC9\xDE",1503,"\xD4\xD9",1489,"\xD1\xD4",1473,"\xC2\xC5",1451,"\xC1\xCB",1413,"\xCC\xD1",1408,"\xCD\xC9",1407,"\xC7\xC1",1375,"\xCF\xDA",1366,"\xC2\xCF",1339,"\xD8\xCE",1333,"\xC5\xCB",1328,"\xC7\xD2",1321,"\xCF\xDE",1313,"\xD0\xC9",1290,"\xD3\xC9",1266,"\xC1\xCD",1261,"\xD0\xCC",1247,"\xC2\xD2",1220,"\xDE\xC1",1186,"\xD3\xC5",1170,"\xD6\xC9",1153,"\xC5\xDA",1151,"\xD3\xCC",1150,"\xC9\xCD",1149,"\xC1\xD0",1145,"\xC9\xD1",1127,"\xCD\xD9",1126,
				"\xCE\xD4",1124,"\xD5\xD3",1115,"\xC3\xC9",1086,"\xD4\xCE",1082,"\xD6\xC5",1081,"\xC2\xC1",1073,"\xD3\xC1",1053,"\xCB\xD5",1030,"\xC2\xC9",1007,"\xC9\xCC",1002,"\xCF\xCA",976,"\xC5\xD0",973,"\xCF\xD6",966,"\xD5\xD2",947,"\xD2\xD1",937,"\xC1\xC7",923,"\xCB\xCC",921,"\xC1\xC2",918,"\xDA\xCE",909,"\xC4\xCE",907,"\xC2\xCC",907,"\xD2\xCE",895,"\xDE\xCE",895,"\xD7\xCC",887,"\xC1\xDE",885,"\xD4\xD5",883,"\xDA\xCF",877,"\xCE\xD1",877,"\xD6\xC1",861,"\xDB\xC5",860,"\xC5\xDE",855,"\xC3\xC5",855,"\xC1\xD6",842,
				"\xC7\xC9",842,"\xC5\xCA",831,"\xC8\xCF",826,"\xD2\xD9",812,"\xD4\xD3",808,"\xD5\xD0",790,"\xDA\xC9",762,"\xDA\xD7",760,"\xCC\xD5",759,"\xD5\xC4",752,"\xD3\xCE",748,"\xC9\xC4",748,"\xDB\xC1",743,"\xD5\xCC",741,"\xD3\xD3",738,"\xC4\xD2",723,"\xC9\xC3",719,"\xC9\xCF",705,"\xD7\xCE",701,"\xD5\xDE",699,"\xC5\xC7",696,"\xD3\xD7",695,"\xC4\xD5",692,"\xD0\xD5",683,"\xC7\xCC",682,"\xD2\xD4",676,"\xDD\xC5",649,"\xD5\xC2",646,"\xC1\xC5",643,"\xD5\xCD",642,"\xD5\xD7",640,"\xDA\xC4",627,"\xC9\xC7",626,"\xCB\xD4",609,
				"\xC3\xC1",605,"\xD5\xDB",602,"\xD3\xD5",589,"\xC5\xD6",584,"\xCC\xC0",584,"\xD4\xCB",578,"\xCE\xD8",567,"\xC2\xD5",564,"\xD3\xCD",562,"\xC8\xC1",561,"\xD9\xD3",551,"\xCF\xC8",550,"\xD8\xC5",545,"\xCD\xD5",543,"\xCC\xD9",532,"\xD5\xCB",529,"\xC6\xC9",529,"\xD7\xD3",527,"\xD7\xCB",525,"\xDA\xCD",525,"\xD5\xD6",524,"\xC1\xC3",523,"\xD1\xDD",519,"\xCE\xD3",518,"\xC4\xD7",514,"\xCE\xCB",511,"\xC9\xD0",500,"\xD3\xD9",499,"\xC5\xC2",497,"\xD1\xCE",495,"\xC1\xC8",495,"\xD5\xC0",487,"\xC5\xDB",486,"\xC1\xDB",485,
				"\xCF\xDB",480,"\xC7\xCE",477,"\xC5\xC3",474,"\xCD\xCE",473,"\xD1\xD7",462,"\xD5\xC7",461,"\xD5\xCE",455,"\xD2\xCD",450,"\xDE\xCB",446,"\xCF\xC9",442,"\xDD\xC1",437,"\xC6\xCF",434,"\xCE\xC4",423,"\xD2\xCB",423,"\xD6\xC4",419,"\xD2\xD3",412,"\xD9\xD4",410,"\xD4\xD1",404,"\xD1\xC0",404,"\xCF\xC5",403,"\xD7\xD2",400,"\xD3\xD8",398,"\xC5\xCF",381,"\xCC\xCB",375,"\xC7\xD5",369,"\xC4\xD9",368,"\xD6\xCE",366,"\xD9\xD0",364,"\xDA\xD9",359,"\xDA\xD2",357,"\xC9\xC2",356,"\xC8\xC9",355,"\xD7\xD1",354,"\xC1\xDD",351,
				"\xD5\xC8",347,"\xC4\xD3",344,"\xDA\xC5",342,"\xC7\xC5",335,"\xDA\xCC",332,"\xD9\xD2",332,"\xD2\xC7",332,"\xDB\xCB",326,"\xCF\xCF",326,"\xC8\xCC",325,"\xD7\xDA",324,"\xDE\xD5",322,"\xD9\xCB",321,"\xCB\xD3",319,"\xC5\xC8",317,"\xC1\xC6",317,"\xCB\xCE",316,"\xD4\xCC",316,"\xDA\xD5",313,"\xD2\xC4",312,"\xCC\xCC",308,"\xD5\xDD",307,"\xDA\xC7",304,"\xC9\xC8",303,"\xD7\xD5",303,"\xC4\xCB",300,"\xD9\xCC",299,"\xC9\xC1",294,"\xDA\xC2",289,"\xD9\xDB",287,"\xD1\xDA",286,"\xD2\xD7",284,"\xCE\xC3",278,"\xD8\xD1",278,
				"\xCE\xDE",274,"\xD8\xCB",273,"\xC1\xCA",273,"\xD5\xDA",271,"\xD1\xC4",270,"\xC9\xDD",270,"\xCF\xC6",269,"\xD2\xD6",268,"\xC6\xC1",265,"\xC9\xD6",263,"\xD0\xD9",263,"\xC5\xDD",262,"\xDB\xCC",261,"\xC1\xC9",257,"\xCA\xCE",256,"\xC8\xD2",252,"\xC8\xCE",251,"\xC2\xCE",250,"\xD1\xD3",249,"\xC6\xC5",243,"\xD7\xD4",241,"\xD2\xD8",237,"\xCD\xD0",236,"\xC9\xC6",235,"\xC2\xD9",235,"\xDB\xCE",233,"\xDE\xD8",229,"\xCF\xD1",226,"\xCD\xCC",225,"\xCF\xDD",225,"\xCA\xDB",225,"\xCB\xC5",219,"\xC4\xCC",216,"\xD1\xC7",211,
				"\xCB\xD7",211,"\xCC\xCE",209,"\xD0\xD4",208,"\xD3\xDE",207,"\xC5\xC5",207,"\xCD\xD1",206,"\xC4\xD1",205,"\xD0\xD1",205,"\xD2\xDB",204,"\xCF\xD5",203,"\xC9\xDB",198,"\xC8\xD7",192,"\xD8\xD4",192,"\xD2\xDE",192,"\xD9\xC7",187,"\xD1\xD6",187,"\xCE\xC7",186,"\xD3\xC8",183,"\xC0\xD4",182,"\xCA\xCB",181,"\xD0\xCE",180,"\xD3\xD2",178,"\xC3\xCF",172,"\xC5\xC9",166,"\xDF\xC5",166,"\xD9\xC8",166,"\xCF\xC3",163,"\xD9\xC4",161,"\xDE\xCF",160,"\xD2\xC2",160,"\xD9\xCE",154,"\xDC\xCB",153,"\xD1\xDE",152,"\xD7\xD0",151,
				"\xC0\xC2",150,"\xD1\xCB",150,"\xD1\xCD",149,"\xD3\xC3",146,"\xC4\xD0",146,"\xC2\xD3",146,"\xDB\xD4",143,"\xD2\xC8",142,"\xD9\xCD",142,"\xD5\xC5",142,"\xC0\xDE",141,"\xC1\xD1",138,"\xD3\xDB",133,"\xD9\xDE",133,"\xD4\xDE",133,"\xC1\xD5",131,"\xC5\xD1",131,"\xD7\xC4",129,"\xD9\xC2",129,"\xC4\xDE",128,"\xC6\xD2",128,"\xD4\xD0",128,"\xCD\xC2",127,"\xD1\xC5",126,"\xC5\xC6",125,"\xDA\xD6",124,"\xDB\xD8",123,"\xC4\xD8",122,"\xC5\xD5",122,"\xD1\xD2",121,"\xD4\xCD",121,"\xDD\xD5",121,"\xD9\xDA",119,"\xD6\xD5",119,
				"\xD2\xD0",118,"\xCD\xCD",118,"\xCD\xCB",117,"\xC5\xC1",117,"\xD3\xC2",116,"\xD6\xCF",113,"\xCB\xC3",112,"\xD8\xCD",111,"\xD6\xCB",111,"\xD4\xD4",108,"\xD2\xCC",108,"\xDA\xD1",108,"\xDC\xCC",106,"\xDA\xCB",105,"\xDB\xD5",105,"\xD1\xC8",104,"\xCC\xD4",103,"\xD2\xC0",102,"\xD0\xD3",101,"\xD2\xDA",101,"\xC6\xCC",100,"\xD3\xC7",100,"\xC8\xD5",99,"\xD0\xD8",99,"\xC2\xDD",98,"\xD5\xC1",96,"\xDE\xD4",96,"\xC4\xCD",95,"\xC0\xD2",94,"\xC0\xC4",93,"\xDB\xCF",90,"\xD4\xC4",90,"\xD2\xD2",89,"\xC8\xD4",89,"\xC2\xCD",89,
				"\xD0\xD0",88,"\xCE\xC6",88,"\xD7\xD8",87,"\xCF\xC0",87,"\xC5\xC0",86,"\xD8\xC0",85,"\xD2\xC3",84,"\xC4\xC3",84,"\xC6\xD5",83,"\xD4\xC2",83,"\xC4\xD6",83,"\xDF\xD1",82,"\xC4\xDB",81,"\xC2\xDF",81,"\xC2\xCB",80,"\xCA\xD4",80,"\xC8\xCD",80,"\xCC\xDA",79,"\xD7\xCD",79,"\xCA\xCD",78,"\xD8\xC3",78,"\xD3\xC4",77,"\xD9\xD6",77,"\xC2\xD7",77,"\xD1\xC3",73,"\xC0\xCE",73,"\xC4\xD4",73,"\xD3\xC6",73,"\xD7\xD7",72,"\xCE\xDA",72,"\xD0\xDE",71,"\xD4\xC0",71,"\xDC\xD4",70,"\xCA\xC4",67,"\xD7\xDE",67,"\xD8\xDA",66,
				"\xDA\xD8",66,"\xD8\xDB",65,"\xC3\xD7",65,"\xCC\xC7",65,"\xC4\xC7",65,"\xD0\xCB",64,"\xD1\xCC",64,"\xC2\xD8",64,"\xC0\xCB",63,"\xD1\xC2",63,"\xDA\xDF",63,"\xCD\xD3",63,"\xC2\xD1",63,"\xCB\xCB",62,"\xCC\xC4",61,"\xCC\xDE",61,"\xC9\xD5",60,"\xCD\xD2",60,"\xD7\xDD",60,"\xCC\xC2",60,"\xD2\xDD",60,"\xDA\xD4",59,"\xC4\xC4",59,"\xDB\xD0",59,"\xD6\xC7",59,"\xD6\xD8",59,"\xC0\xD3",58,"\xD6\xCD",58,"\xC4\xC2",56,"\xDC\xD2",56,"\xCF\xC1",55,"\xC0\xCD",55,"\xC6\xD4",55,"\xCE\xC0",54,"\xD8\xDD",54,"\xD4\xC7",53,
				"\xD8\xDE",51,"\xC0\xDA",51,"\xC2\xC4",50,"\xCC\xD6",49,"\xD5\xC9",49,"\xD6\xD6",49,"\xD8\xC4",49,"\xCD\xDE",49,"\xCE\xD7",48,"\xCC\xD3",48,"\xDB\xD7",48,"\xCB\xDB",48,"\xC2\xD6",48,"\xD1\xD0",48,"\xDD\xCE",47,"\xCF\xDC",47,"\xC4\xDA",46,"\xD1\xCA",45,"\xD2\xC6",44,"\xC1\xCF",44,"\xD5\xC6",44,"\xDC\xD0",43,"\xD9\xDD",43,"\xC0\xC8",42,"\xD8\xC9",41,"\xC8\xD3",41,"\xC0\xCC",41,"\xD5\xCA",41,"\xC4\xC0",40,"\xD3\xDF",40,"\xC0\xDB",40,"\xDC\xCD",40,"\xC2\xD4",39,"\xDE\xCC",39,"\xDC\xCE",39,"\xCA\xDE",38,
				"\xD4\xDB",38,"\xD3\xD6",38,"\xDC\xD3",38,"\xC3\xCB",38,"\xC7\xDE",37,"\xCD\xD8",36,"\xC3\xD9",36,"\xD7\xC3",36,"\xC1\xDC",35,"\xD5\xD1",35,"\xDA\xDA",35,"\xC7\xDB",34,"\xCE\xDD",34,"\xD8\xD7",34,"\xC7\xCD",34,"\xD8\xC2",34,"\xCE\xD6",34,"\xCD\xC6",33,"\xC7\xD7",33,"\xCA\xC3",33,"\xC7\xCB",32,"\xC9\xC9",32,"\xC8\xDB",32,"\xDA\xDB",31,"\xC4\xDF",31,"\xCC\xD7",31,"\xDA\xDE",31,"\xC8\xC5",31,"\xC2\xC0",30,"\xC2\xC8",30,"\xC0\xD6",30,"\xC6\xC6",29,"\xCA\xCC",28,"\xD6\xD2",28,"\xD7\xC2",28,"\xD5\xC3",28,
				"\xD7\xC7",27,"\xCB\xDA",27,"\xD4\xC8",27,"\xDB\xCD",26,"\xCE\xDB",26,"\xD3\xC0",26,"\xD8\xC7",26,"\xD4\xDD",25,"\xCE\xD2",25,"\xD8\xCF",25,"\xCC\xCD",24,"\xD4\xC3",24,"\xC2\xC3",23,"\xC0\xC7",23,"\xCC\xD0",22,"\xDE\xD2",22,"\xD5\xCF",22,"\xD8\xC6",22,"\xCE\xC8",22,"\xC2\xC2",22,"\xC2\xDE",22,"\xC9\xC0",21,"\xC8\xDE",21,"\xD1\xDB",21,"\xD9\xD1",20,"\xCD\xC3",20,"\xC2\xDA",20,"\xC7\xD3",20,"\xC0\xC3",20,"\xDC\xD7",19,"\xD0\xC3",19,"\xCA\xCF",19,"\xC2\xDB",18,"\xC4\xC8",18,"\xC6\xD3",17,"\xD0\xDB",17,
				"\xC8\xCB",17,"\xC6\xCD",17,"\xCD\xDD",17,"\xDE\xCD",17,"\xD7\xDF",16,"\xC8\xC7",16,"\xC7\xC4",16,"\xDC\xC6",16,"\xD4\xDA",16,"\xC6\xCE",16,"\xD4\xDF",16,"\xC2\xC7",16,"\xC8\xD0",15,"\xD6\xC2",15,"\xCD\xD7",15,"\xDE\xDB",15,"\xDD\xD2",15,"\xC0\xCA",14,"\xD7\xC8",14,"\xD8\xD0",14,"\xDE\xD7",14,"\xD9\xC5",14,"\xCA\xD2",13,"\xC3\xD5",13,"\xD4\xC6",13,"\xD7\xD6",13,"\xD9\xC3",12,"\xD4\xD6",12,"\xD6\xCC",12,"\xCA\xC2",12,"\xCD\xD4",12,"\xDF\xC0",11,"\xCD\xC7",11,"\xC0\xD0",11,"\xDA\xC0",11,"\xDB\xC0",11,
				"\xD5\xDC",11,"\xCA\xC6",11,"\xD1\xC9",11,"\xCA\xDD",11,"\xD9\xD5",10,"\xCD\xDB",10,"\xD9\xC9",10,"\xC6\xD9",10,"\xD3\xDD",9,"\xD1\xD5",9,"\xDC\xC7",9,"\xCD\xC8",9,"\xDD\xCF",9,"\xC6\xD8",8,"\xCE\xDF",8,"\xCA\xD7",8,"\xC0\xD7",8,"\xCC\xC8",8,"\xC8\xC4",8,"\xCD\xDA",8,"\xCB\xC0",8,"\xD0\xC6",8,"\xCC\xDB",7,"\xDA\xC3",7,"\xDC\xC4",7,"\xCA\xC8",7,"\xC9\xDC",7,"\xCC\xDD",7,"\xDB\xD2",7,"\xCA\xDA",7,"\xD8\xC8",7,"\xC3\xCD",7,"\xD5\xD5",6,"\xCC\xC6",6,"\xCA\xD0",6,"\xCA\xC5",6,"\xDD\xD8",6,"\xC0\xC1",6,
				"\xCE\xD0",6,"\xD6\xD3",6,"\xCB\xCD",6,"\xCB\xC4",6,"\xC0\xC0",5,"\xC1\xC1",5,"\xC7\xC2",5,"\xC0\xCF",5,"\xD3\xDC",5,"\xDA\xD3",5,"\xC6\xD1",5,"\xC0\xC9",5,"\xC0\xC6",5,"\xCE\xCC",5,"\xD6\xD7",5,"\xC6\xC0",5,"\xCB\xDE",4,"\xDC\xCA",4,"\xD1\xD1",4,"\xCE\xC2",4,"\xD3\xDA",4,"\xC3\xD4",4,"\xCE\xDC",4,"\xD0\xC0",4,"\xC3\xD3",4,"\xC6\xC7",4,"\xC3\xCC",4,"\xC2\xD0",4,"\xD0\xDD",4,"\xD6\xC3",4,"\xC8\xDA",4,"\xC8\xDF",4,"\xD8\xD6",3,"\xC8\xC3",3,"\xC4\xC6",3,"\xC4\xDC",3,"\xC8\xDC",3,"\xC7\xD4",3,"\xDD\xD7",3,
				"\xC6\xC2",3,"\xCC\xC3",3,"\xC3\xDB",3,"\xC5\xDC",3,"\xDC\xC8",3,"\xCB\xC8",3,"\xCD\xC4",3,"\xCD\xDC",3,"\xCA\xC7",3,"\xCB\xC7",2,"\xC8\xC2",2,"\xCE\xCD",2,"\xC7\xDA",2,"\xCD\xD6",2,"\xDB\xC3",2,"\xDE\xC8",2,"\xDC\xC2",2,"\xCA\xC9",2,"\xC6\xDE",2,"\xD7\xC6",2,"\xD7\xC0",2,"\xD6\xDE",2,"\xD2\xDC",2,"\xD6\xDA",2,"\xD6\xD0",2,"\xDC\xDA",2,"\xDC\xDB",2,"\xC3\xC7",2,"\xCD\xC0",2,"\xCB\xC6",2,"\xC3\xCE",2,"\xD0\xCD",2,"\xDB\xC2",2,"\xC3\xD0",2,"\xD4\xDC",2,"\xCA\xD1",2,"\xDB\xC8",1,"\xC7\xDC",1,"\xC7\xD0",1,
				"\xC7\xC7",1,"\xDB\xDE",1,"\xC7\xC0",1,"\xC6\xDB",1,"\xDC\xCF",1,"\xC6\xD7",1,"\xDC\xD1",1,"\xC6\xCB",1,"\xC6\xC4",1,"\xDD\xCD",1,"\xD8\xDC",1,"\xD0\xC4",1,"\xC2\xC6",1,"\xCB\xD6",1,"\xCB\xD0",1,"\xDE\xD3",1,"\xDA\xD0",1,"\xC0\xC5",1,"\xC3\xC4",1,"\xCB\xD9",1,"\xCC\xD2",1,"\xC0\xDC",1,"\xC8\xD6",1,"\xCB\xC2",1,"\xC4\xDD",1,"\xC2\xDC",1,"\xDE\xC4",1,"\xD6\xC0",1,"\xC3\xD2",1,"\xDA\xDC",1,"\xD6\xD4",1,"\xDE\xDE",1,"\xC3\xC3",1,"\xC8\xC6",1,"\xC3\xDA",1,"\xD0\xC8",1
			);

sub cp_detect {
	my ( $package, $test_string, $debug, @cp ) = @_;
	my ( $string, $code_page, $key, %results1, %results2 );
	local $_;

	unless ( defined $test_string and length($test_string) > 0 ) {
		if ( $^W ) {
			warn 'Test string not defined';
		}
		return undef;
	}
	unless ( defined $debug ) {
		$debug = 0;
	}
	unless ( scalar @cp ) {
		@cp = qw( KOI8-R CP1251 IBM866 );
#		@cp = qw( UTF8 KOI8-R CP1251 IBM866 ISO-8859-5 MAC-UK IBM855 );
	}

	foreach $code_page ( @cp ) {
		if ( defined $code_page and $code_page eq 'UTF8' ) {
#
#     UTF8     
# results2...,    
#  result1{...}    result1{UTF8} ..    
#     result2...
#
			my $ind	= -1;
			$results1{$code_page} = 0;
			$results2{$code_page} = 0;
			{
				if ( ( $ind = index( $test_string, "\xd0", $ind+1 ) ) > -1 ) {
					if ( substr( $test_string, $ind+1, 1 ) =~ m/^[\x81\x90-\xbf]$/ ) {
						$results2{$code_page}++;
					}
					redo;
				}
			}
			$ind = -1;
			{
				if ( ( $ind = index( $test_string, "\xd1", $ind+1 ) ) > -1 ) {
					if ( substr( $test_string, $ind+1, 1 ) =~ m/^[\x80-\x91]$/ ) {
						$results2{$code_page}++;
					}
					redo;
				}
			}
			next;
		} elsif ( not defined $code_page or not exists $CHECK{$code_page} ) {
			if ( $^W ) {
				unless ( defined $code_page ) {
					warn 'Undefined codepage variable';
				} else {
					warn "Unknown codepage `$code_page'";
				}
			}
			next;
		}

		$string = $test_string;
#
#      KOI8-R
#
		$CHECK{$code_page}->($string);
#
#    
#
		$string =~ tr/\xE1\xE2\xF7\xE7\xE4\xE5\xB3\xF6\xFA\xE9\xEA\xEB\xEC\xED\xEE\xEF\xF0\xF2\xF3\xF4\xF5\xE6\xE8\xE3\xFE\xFB\xFD\xFF\xF9\xF8\xFC\xE0\xF1/\xC1\xC2\xD7\xC7\xC4\xC5\xA3\xD6\xDA\xC9\xCA\xCB\xCC\xCD\xCE\xCF\xD0\xD2\xD3\xD4\xD5\xC6\xC8\xC3\xDE\xDB\xDD\xDF\xD9\xD8\xDC\xC0\xD1/;
#
#    -
#
		$string =~ tr/\xC1\xC2\xD7\xC7\xC4\xC5\xA3\xD6\xDA\xC9\xCA\xCB\xCC\xCD\xCE\xCF\xD0\xD2\xD3\xD4\xD5\xC6\xC8\xC3\xDE\xDB\xDD\xDF\xD9\xD8\xDC\xC0\xD1/ /cs;
#
#    3  (  ... ).
#
		if ( $COMPRESS_STR ) {
			$string =~ s/(.)\1\1+/$1$1/g;
		}
#
#      %results2
#      %results1   
#
		$results1{$code_page} = 0;
		$results2{$code_page} = 0;

		for $key ( 0 .. length($string) - 2 ) {
			if ( exists $TABLE{ substr $string, $key, 2 } ) {
				$results1{$code_page} += $TABLE{ substr $string, $key, 2 };
				$results2{$code_page} ++;
			}
		}
	}
#
#         ???
#
	unless ( keys %results1 ) {
		if ( $^W ) {
			unless ( defined $code_page ) {
				warn 'Undefined codepages variable';
			}
		}
		return undef;
	}
#
#    ,   UTF8  
#  ...
#
	if ( exists $results2{'UTF8'} and $results2{'UTF8'} > 0 ) {
		$results1{'UTF8'} =	( sort { $b <=> $a } values %results1 )[0];
	}
#
#       
#  $DETECT_TYPE
#
	if ( $DETECT_TYPE == 1 ) {
		( $key ) = ( sort {
							$results2{$b} <=> $results2{$a} or
							$results1{$b} <=> $results1{$a}
						} keys %results2 )[0];
	} else {
		( $key ) = ( sort {
							$results1{$b} <=> $results1{$a} or
							$results2{$b} <=> $results2{$a}
						} keys %results1 )[0];
	}

	if ( $debug ) {
		foreach $code_page ( sort { $a cmp $b } keys %results2 ) {
			printf STDERR "CP: %-10s : %10d : %15d\n",	$code_page,
														$results2{$code_page},
														$results1{$code_page};
		}
	}
#
#  results2{$key} == 0      ...
#
	if ( $results2{$key} == 0 ) {
		return undef;
	} else {
		return $key;
	}
}

1;

=pod

=head1 NAME

Cyrillic::CPdetect -      .

=head1 SYNOPSIS

 use Cyrillic::CPdetect;

 $cp = Cyrillic::CPdetect->cp_detect( $test_string, $debug, @cp );

    undef    .

=over 15

=item $test_string

    .

=item $debug

1  0.  1    .

=item @cp

      $test_string.  : KOI8-R, CP1251, IBM866. : KOI8-R, CP1251, IBM866, ISO-8859-5, MAC-UK, IBM855, UTF8 (   , ..  ISO-8859-5   ,     2    ... ).

=back

=head1 DESCRIPTION

       $Cyrillic::CPdetect::DETECT_TYPE,      1           %TABLE,      2              %TABLE.   $DETECT_TYPE = 2

  : $Cyrillic::CPdetect::COMPRESS_STR -  1 (   )           (  ) ''  2 (  ),      .     ~ 10%    ,      .

=head1 AUTHOR

 Okunev Igor V.  mailto:igor@prv.mts-nn.ru
                 http://www.mts-nn.ru/~gosha


=head1 REVISION

$Id: CPdetect.pm,v 1.14 2002/12/02 18:51:37 gosha Exp $

=cut

